<?php

// Helper function for Base64 URL encoding
function base64UrlEncode($text) {
    return str_replace(
        ['+', '/', '='],
        ['-', '_', ''],
        base64_encode($text)
    );
}

function getAccessToken($authConfigFile, $retryCount = 0) {
    $maxRetries = 3;
    $authConfigString = file_get_contents($authConfigFile);

    if (!$authConfigString) {
        die("Failed to read the service account file: $authConfigFile.");
    }

    $authConfig = json_decode($authConfigString);
    if (!$authConfig || !isset($authConfig->private_key)) {
        die("Invalid service account details.");
    }

    $secret = openssl_get_privatekey($authConfig->private_key);
    if (!$secret) {
        die("Failed to load the private key.");
    }

    $header = json_encode(['typ' => 'JWT', 'alg' => 'RS256']);
    $time = time();
    $start = $time - 60;
    $end = $start + 3600; // 1-hour token expiration

    $payload = json_encode([
        "iss" => $authConfig->client_email,
        "scope" => "https://www.googleapis.com/auth/firebase.messaging",
        "aud" => "https://oauth2.googleapis.com/token",
        "exp" => $end,
        "iat" => $start
    ]);

    $base64UrlHeader = base64UrlEncode($header);
    $base64UrlPayload = base64UrlEncode($payload);
    openssl_sign($base64UrlHeader . "." . $base64UrlPayload, $signature, $secret, OPENSSL_ALGO_SHA256);
    openssl_free_key($secret);

    $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . base64UrlEncode($signature);

    // Use cURL instead of file_get_contents
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://oauth2.googleapis.com/token");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=urn:ietf:params:oauth:grant-type:jwt-bearer&assertion=' . $jwt);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/x-www-form-urlencoded"]);
    $responseText = curl_exec($ch);

    if (curl_errno($ch)) {
        echo "Failed to retrieve the token. cURL Error: " . curl_error($ch) . "\n";
        curl_close($ch);
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }

    curl_close($ch);

    $response = json_decode($responseText);

    if (isset($response->access_token)) {
        return $response->access_token;
    } else {
        echo "Failed to get the access token. Response: " . $responseText . "\n";
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }
}

function sendFirebaseTopicNotification($accessToken, $projectId, $topic, $title, $body, $customData = []) {
    $url = "https://fcm.googleapis.com/v1/projects/$projectId/messages:send";
    $message = [
        "message" => [
            "topic" => $topic,
            "data" => array_merge($customData, [
                "title" => $title,
                "body" => $body,
                "click_action" => "OPEN_SPECIFIC_PAGE", // Custom click action
                "target_page" => "DetailsActivity",      // Example of a target page or activity
                "extra_param" => "example_param_value"   // Additional data if needed
            ])
        ]
    ];

    $messageJson = json_encode($message);

    $headers = [
        "Authorization: Bearer $accessToken",
        "Content-Type: application/json"
    ];

    $options = [
        'http' => [
            'method' => 'POST',
            'header' => implode("\r\n", $headers),
            'content' => $messageJson,
        ]
    ];

    $context = stream_context_create($options);
    $response = file_get_contents($url, false, $context);

    if ($response === false) {
        return "Failed to send the push notification.";
    } else {
        return $response;
    }
}

// Function to save data to Firebase Realtime Database for a specific project
function saveToFirebase($projectId, $title, $body, $featureImage, $externalLink) {
    $firebaseUrl = "https://$projectId-default-rtdb.firebaseio.com/notifications/bangla.json";

    $data = [
        'title' => $title,
        'body' => $body,
        'feature_image' => $featureImage,
        'external_link' => $externalLink,
        'timestamp' => time()
    ];

    $options = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/json\r\n",
            'content' => json_encode($data)
        ]
    ];

    $context = stream_context_create($options);
    $response = file_get_contents($firebaseUrl, false, $context);

    if ($response === false) {
        return "Failed to save data to Firebase.";
    } else {
        return "Data saved successfully.";
    }
}

// Process form data
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Firebase project configurations
    $projects = [
          //islamicapps.storage2@gmail.com
          'bangla-storage-2-1' => 'pvkeys/bangla-storage-2-1.json', 
          'bangla-storage-2-2' => 'pvkeys/bangla-storage-2-2.json', 
          'bangla-storage-2-3' => 'pvkeys/bangla-storage-2-3.json', 
          'bangla-storage-2-4' => 'pvkeys/bangla-storage-2-4.json', 
          'bangla-storage-2-5' => 'pvkeys/bangla-storage-2-5.json', 
          'bangla-storage-2-6' => 'pvkeys/bangla-storage-2-6.json', 
          'bangla-storage-2-7' => 'pvkeys/bangla-storage-2-7.json', 
          'bangla-storage-2-7-b93d3' => 'pvkeys/bangla-storage-2-7-b93d3.json', 
          'bangla-storage-2-8' => 'pvkeys/bangla-storage-2-8.json', 
          'bangla-storage-2-9' => 'pvkeys/bangla-storage-2-9.json', 
          'bangla-storage-2-10' => 'pvkeys/bangla-storage-2-10.json', 
          //   '' => 'pvkeys/.json', 
          //   '' => 'pvkeys/.json', 
         // islamicappsstore.info@gmail.com
        'islamic-apps-94b4d' => 'pvkeys/islamic-apps-1.json', 
        'islamic-apps-2' => 'pvkeys/islamic-apps-2.json',    
        'islamic-apps---iasi-3' => 'pvkeys/islamic-apps---iasi-3.json',
        'islamic-apps---iasi-4' => 'pvkeys/islamic-apps---iasi-4.json',
        'islamic-apps---iasi-5' => 'pvkeys/islamic-apps---iasi-5.json',
        'islamic-apps---iasi-6' => 'pvkeys/islamic-apps---iasi-6.json',
        'islamic-apps---iasi-7' => 'pvkeys/islamic-apps---iasi-7.json',
        'islamic-apps---iasi-8' => 'pvkeys/islamic-apps---iasi-8.json',
        'islamic-apps---iasi-9' => 'pvkeys/islamic-apps---iasi-9.json',
        'islamic-apps---iasi-9' => 'pvkeys/islamic-apps---iasi-9.json',
        'islamic-apps---iasi-10' => 'pvkeys/islamic-apps---iasi-10.json',
        
        
        
        // islamicapps.firebase1@gmail.com // bangla apps
        
        'bangla1-alt' => 'pvkeys/bangla1-alt.json',  // islamicapps.firebase1@gmail.com // bangla apps
        'bangla2-alt' => 'pvkeys/bangla2-alt.json',
        'bangla3-alt' => 'pvkeys/bangla3-alt.json',
        'bangla4-alt' => 'pvkeys/bangla4-alt.json',
        'bangla5-alt' => 'pvkeys/bangla5-alt.json',
        'bangla6-alt' => 'pvkeys/bangla6-alt.json',
        'bangla-17-alt' => 'pvkeys/bangla-17-alt.json',
        'bangla-18-alt' => 'pvkeys/bangla-18-alt.json',
        'bangla-19-alt' => 'pvkeys/bangla-19-alt.json',
        'bangla-20-alt' => 'pvkeys/bangla-20-alt.json',
        
        
        // islamicapps.firebase5@gmail.com //
        
        'bangla-7-alt' => 'pvkeys/bangla-7-alt.json', 
        'bangla-8-alt' => 'pvkeys/bangla-8-alt.json',
        'bangla-9-alt' => 'pvkeys/bangla-9-alt.json',
        'bnagla-10-alt' => 'pvkeys/bnagla-10-alt.json',
        'bangla-11-alt' => 'pvkeys/bangla-11-alt.json',
        'bangla-12-alt' => 'pvkeys/bangla-12-alt.json',
        'bangla-13-alt' => 'pvkeys/bangla-13-alt.json',
        'bangla-14-alt' => 'pvkeys/bangla-14-alt.json',
        'bangla-15-alt' => 'pvkeys/bangla-15-alt.json',
        'bangla-16-alt' => 'pvkeys/bangla-16-alt.json',

        
        //arifhasnat.business@gmail.com

        //'hoxro-android' => 'pvkeys/hoxro-android.json',
        //'islamic-apps---english1' => 'pvkeys/islamic-apps---english1.json', // english - tafsir ibn kathir english
        
        //arifhasnat.business2@gmail.com

        
        'bangla1-b8ffe' => 'pvkeys/bangla1-b8ffe.json',
        'bangla2-a384a' => 'pvkeys/bangla2-a384a.json',
        'bangla3-fbc00' => 'pvkeys/bangla3-fbc00.json',
        
        
        'bangla4-65441' => 'pvkeys/bangla4-65441.json',
        'bangla5-6ac6b' => 'pvkeys/bangla5-6ac6b.json',
        'bangla6-441df' => 'pvkeys/bangla6-441df.json',
        'bangla7-2aa86' => 'pvkeys/bangla7-2aa86.json',
       
        
        //arifhasnat.business3@gmail.com

        'bangla-15' => 'pvkeys/bangla-15.json',
        'bangla-16' => 'pvkeys/bangla-16.json',
        'bangla-17' => 'pvkeys/bangla-17.json',
        'bangla11' => 'pvkeys/bangla11.json',

        
        //arifhasnat.business4@gmail.com
        
        'bangla41-96d7e' => 'pvkeys/bangla41-96d7e.json',
        'bangla42-5d6c3' => 'pvkeys/bangla42-5d6c3.json',
        'bangla43-2c116' => 'pvkeys/bangla43-2c116.json',
        
        
        'bangla44-77ce7' => 'pvkeys/bangla44-77ce7.json',
        'bangla45-851f5' => 'pvkeys/bangla45-851f5.json',
        'bangla46-30d11' => 'pvkeys/bangla46-30d11.json',
    ];

    // Form inputs
    $title = $_POST['title'];
    $body = $_POST['body'];
    $featureImage = $_POST['feature_image'];
    $externalLink = $_POST['external_link'];
    $topic = 'notification_new';  // Firebase topic

    // Custom data for notification
    $customData = [
        'title' => $title,
        'body' => $body,
        'feature_image' => $featureImage,
        'external_link' => $externalLink
    ];

    // Check which button was clicked
    $action = $_POST['action'];
    $logOutput = "";

    foreach ($projects as $projectId => $authConfigFile) {
        if ($action === "Save to Firebase") {
            // Save data to Firebase Realtime Database
            $saveResponse = saveToFirebase($projectId, $title, $body, $featureImage, $externalLink);
            $logOutput .= "Project: $projectId <br>Save Response: $saveResponse<br><br>";
        } elseif ($action === "Send Push Notification") {
            // Get OAuth2 access token
            $accessToken = getAccessToken($authConfigFile);
            // Send push notification to the topic
            $notificationResponse = sendFirebaseTopicNotification($accessToken, $projectId, $topic, $title, $body, $customData);
            $logOutput .= "Project: $projectId <br>Notification Response: $notificationResponse<br><br>";
        }
    }

    // Display success message with hidden log content
    echo "<html>
        <head>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    background-color: #f4f4f4;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    min-height: 100vh;
                    margin: 0;
                    padding: 10px;
                    box-sizing: border-box;
                }

                .container {
                    text-align: center;
                    background-color: white;
                    padding: 30px;
                    border-radius: 10px;
                    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
                    max-width: 100%;
                    width: 100%;
                    box-sizing: border-box;
                }

                @media(min-width: 768px) {
                    .container {
                        max-width: 600px;
                        padding: 50px;
                    }
                }

                .success-message {
                    font-size: 20px;
                    color: #28a745;
                    margin-bottom: 20px;
                }

                .button {
                    background-color: #007bff;
                    color: white;
                    padding: 10px 20px;
                    font-size: 16px;
                    border: none;
                    border-radius: 5px;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: background-color 0.3s ease;
                }

                .button:hover {
                    background-color: #0056b3;
                }

                .log-section {
                    margin-top: 20px;
                    display: none; /* Hidden by default */
                    max-width: 100%;
                    height: 250px;
                    overflow-y: auto;
                    border: 1px solid #ddd;
                    padding: 15px;
                    background-color: #f9f9f9;
                    text-align: left;
                    white-space: pre-wrap;
                    word-wrap: break-word;
                }

                .toggle-button {
                    background-color: #6c757d;
                    color: white;
                    padding: 8px 16px;
                    font-size: 14px;
                    border: none;
                    border-radius: 4px;
                    cursor: pointer;
                    display: inline-block;
                    margin-top: 10px;
                }

                .toggle-button:hover {
                    background-color: #5a6268;
                }
            </style>
            <script>
                function toggleLogSection() {
                    var logSection = document.getElementById('logSection');
                    if (logSection.style.display === 'none') {
                        logSection.style.display = 'block';
                    } else {
                        logSection.style.display = 'none';
                    }
                }
            </script>
        </head>
        <body>
            <div class='container'>
                <div class='success-message'> Action: $action was processed successfully!</div>
                <button class='toggle-button' onclick='toggleLogSection()'>View Log</button>
                <div class='log-section' id='logSection'>{$logOutput}</div>
                <a class='button' href='index.php'>Back</a>
            </div>
        </body>
    </html>";
}
?>

