<?php

// Helper function for Base64 URL encoding
function base64UrlEncode($text) {
    return str_replace(
        ['+', '/', '='],
        ['-', '_', ''],
        base64_encode($text)
    );
}

function getAccessToken($authConfigFile, $retryCount = 0) {
    $maxRetries = 3;
    $authConfigString = file_get_contents($authConfigFile);

    if (!$authConfigString) {
        die("Failed to read the service account file: $authConfigFile.");
    }

    $authConfig = json_decode($authConfigString);
    if (!$authConfig || !isset($authConfig->private_key)) {
        die("Invalid service account details.");
    }

    $secret = openssl_get_privatekey($authConfig->private_key);
    if (!$secret) {
        die("Failed to load the private key.");
    }

    $header = json_encode(['typ' => 'JWT', 'alg' => 'RS256']);
    $time = time();
    $start = $time - 60;
    $end = $start + 3600; // 1-hour token expiration

    $payload = json_encode([
        "iss"   => $authConfig->client_email,
        "scope" => "https://www.googleapis.com/auth/firebase.messaging",
        "aud"   => "https://oauth2.googleapis.com/token",
        "exp"   => $end,
        "iat"   => $start
    ]);

    $base64UrlHeader  = base64UrlEncode($header);
    $base64UrlPayload = base64UrlEncode($payload);
    openssl_sign($base64UrlHeader . "." . $base64UrlPayload, $signature, $secret, OPENSSL_ALGO_SHA256);
    openssl_free_key($secret);

    $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . base64UrlEncode($signature);

    // Use cURL instead of file_get_contents
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://oauth2.googleapis.com/token");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=urn:ietf:params:oauth:grant-type:jwt-bearer&assertion=' . $jwt);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/x-www-form-urlencoded"]);
    $responseText = curl_exec($ch);

    if (curl_errno($ch)) {
        echo "Failed to retrieve the token. cURL Error: " . curl_error($ch) . "\n";
        curl_close($ch);
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }

    curl_close($ch);

    $response = json_decode($responseText);

    if (isset($response->access_token)) {
        return $response->access_token;
    } else {
        echo "Failed to get the access token. Response: " . $responseText . "\n";
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }
}

/**
 * DATA-ONLY FCM message – lets your FirebaseMessagingService
 * handle click and open specific Activity.
 */
function sendFirebaseTopicNotification($accessToken, $projectId, $topic, $title, $body, $customData = []) {
    $url = "https://fcm.googleapis.com/v1/projects/$projectId/messages:send";

    $message = [
        "message" => [
            "topic" => $topic,
            "data"  => array_merge($customData, [
                "title"        => $title,   // truncated title
                "body"         => $body,    // truncated body
                "click_action" => "OPEN_SPECIFIC_PAGE",
                "target_page"  => "DetailsActivity",
                "extra_param"  => "example_param_value"
            ])
        ]
    ];

    $messageJson = json_encode($message);

    $headers = [
        "Authorization: Bearer $accessToken",
        "Content-Type: application/json"
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $messageJson);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    // Optional: small timeout so one bad request doesn't hang too long
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);

    $response = curl_exec($ch);

    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        // You can also write this into your own logOutput string if you want.
        return "Failed to send the push notification. cURL error: " . $err;
    }

    curl_close($ch);
    return $response;
}


function saveToFirebase($projectId, $title, $body, $featureImage, $externalLink) {
    $firebaseUrl = "https://$projectId-default-rtdb.firebaseio.com/notifications/bangla.json";

    $data = [
        'title'         => $title,
        'body'          => $body,
        'feature_image' => $featureImage,
        'external_link' => $externalLink,
        'timestamp'     => time()
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $firebaseUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json"
    ]);

    // Optional timeout
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);

    $response = curl_exec($ch);

    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        return "Failed to save data to Firebase. cURL error: " . $err;
    }

    curl_close($ch);
    return "Data saved successfully.";
}


/**
 * Rotate through promo URL sets (external_link + feature_image)
 * Stores current index in a small text file so it persists across requests.
 */
function getNextPromo($promoSets, $indexFile = 'promo_index_bangla.txt') {
    $count = count($promoSets);
    if ($count === 0) {
        return [
            'external_link' => '',
            'feature_image' => ''
        ];
    }

    $currentIndex = 0;

    if (file_exists($indexFile)) {
        $content = trim(file_get_contents($indexFile));
        if (is_numeric($content)) {
            $currentIndex = (int)$content;
        }
    }

    // Make sure index is in range
    $currentIndex = $currentIndex % $count;
    $promo        = $promoSets[$currentIndex];

    // Calculate next index and save it
    $nextIndex = ($currentIndex + 1) % $count;
    file_put_contents($indexFile, (string)$nextIndex, LOCK_EX);

    return $promo;
}

// Process form data
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Firebase project configurations
    $projects = [
          //islamicapps.storage2@gmail.com
          'bangla-storage-2-1' => 'pvkeys/bangla-storage-2-1.json', 
          'bangla-storage-2-2' => 'pvkeys/bangla-storage-2-2.json', 
          'bangla-storage-2-3' => 'pvkeys/bangla-storage-2-3.json', 
          'bangla-storage-2-4' => 'pvkeys/bangla-storage-2-4.json', 
          'bangla-storage-2-5' => 'pvkeys/bangla-storage-2-5.json', 
          'bangla-storage-2-6' => 'pvkeys/bangla-storage-2-6.json', 
          'bangla-storage-2-7' => 'pvkeys/bangla-storage-2-7.json', 
          'bangla-storage-2-7-b93d3' => 'pvkeys/bangla-storage-2-7-b93d3.json', 
          'bangla-storage-2-8' => 'pvkeys/bangla-storage-2-8.json', 
          'bangla-storage-2-9' => 'pvkeys/bangla-storage-2-9.json', 
          'bangla-storage-2-10' => 'pvkeys/bangla-storage-2-10.json', 
         // islamicappsstore.info@gmail.com
        'islamic-apps-94b4d' => 'pvkeys/islamic-apps-1.json', 
        'islamic-apps-2' => 'pvkeys/islamic-apps-2.json',    
        'islamic-apps---iasi-3' => 'pvkeys/islamic-apps---iasi-3.json',
        'islamic-apps---iasi-4' => 'pvkeys/islamic-apps---iasi-4.json',
        'islamic-apps---iasi-5' => 'pvkeys/islamic-apps---iasi-5.json',
        'islamic-apps---iasi-6' => 'pvkeys/islamic-apps---iasi-6.json',
        'islamic-apps---iasi-7' => 'pvkeys/islamic-apps---iasi-7.json',
        'islamic-apps---iasi-8' => 'pvkeys/islamic-apps---iasi-8.json',
        'islamic-apps---iasi-9' => 'pvkeys/islamic-apps---iasi-9.json',
        'islamic-apps---iasi-10' => 'pvkeys/islamic-apps---iasi-10.json',
        
        // islamicapps.firebase1@gmail.com // bangla apps
        'bangla1-alt' => 'pvkeys/bangla1-alt.json',
        'bangla2-alt' => 'pvkeys/bangla2-alt.json',
        'bangla3-alt' => 'pvkeys/bangla3-alt.json',
        'bangla4-alt' => 'pvkeys/bangla4-alt.json',
        'bangla5-alt' => 'pvkeys/bangla5-alt.json',
        'bangla6-alt' => 'pvkeys/bangla6-alt.json',
        'bangla-17-alt' => 'pvkeys/bangla-17-alt.json',
        'bangla-18-alt' => 'pvkeys/bangla-18-alt.json',
        'bangla-19-alt' => 'pvkeys/bangla-19-alt.json',
        'bangla-20-alt' => 'pvkeys/bangla-20-alt.json',
        
        // islamicapps.firebase5@gmail.com
        'bangla-7-alt' => 'pvkeys/bangla-7-alt.json', 
        'bangla-8-alt' => 'pvkeys/bangla-8-alt.json',
        'bangla-9-alt' => 'pvkeys/bangla-9-alt.json',
        'bnagla-10-alt' => 'pvkeys/bnagla-10-alt.json',
        'bangla-11-alt' => 'pvkeys/bangla-11-alt.json',
        'bangla-12-alt' => 'pvkeys/bangla-12-alt.json',
        'bangla-13-alt' => 'pvkeys/bangla-13-alt.json',
        'bangla-14-alt' => 'pvkeys/bangla-14-alt.json',
        'bangla-15-alt' => 'pvkeys/bangla-15-alt.json',
        'bangla-16-alt' => 'pvkeys/bangla-16-alt.json',

        // arifhasnat.business2@gmail.com
        'bangla1-b8ffe' => 'pvkeys/bangla1-b8ffe.json',
        'bangla2-a384a' => 'pvkeys/bangla2-a384a.json',
        'bangla3-fbc00' => 'pvkeys/bangla3-fbc00.json',
        'bangla4-65441' => 'pvkeys/bangla4-65441.json',
        'bangla5-6ac6b' => 'pvkeys/bangla5-6ac6b.json',
        'bangla6-441df' => 'pvkeys/bangla6-441df.json',
        'bangla7-2aa86' => 'pvkeys/bangla7-2aa86.json',

        // arifhasnat.business3@gmail.com
        'bangla-15' => 'pvkeys/bangla-15.json',
        'bangla-16' => 'pvkeys/bangla-16.json',
        'bangla-17' => 'pvkeys/bangla-17.json',
        'bangla11' => 'pvkeys/bangla11.json',

        // arifhasnat.business4@gmail.com
        'bangla41-96d7e' => 'pvkeys/bangla41-96d7e.json',
        'bangla42-5d6c3' => 'pvkeys/bangla42-5d6c3.json',
        'bangla43-2c116' => 'pvkeys/bangla43-2c116.json',
        'bangla44-77ce7' => 'pvkeys/bangla44-77ce7.json',
        'bangla45-851f5' => 'pvkeys/bangla45-851f5.json',
        'bangla46-30d11' => 'pvkeys/bangla46-30d11.json',
        
        // shamimanasrinsetu.info@gmail.com
        'si-bangla-1' => 'pvkeys/shamimanasrinsetu_info/si-bangla-1.json',
        'si-bangla-2' => 'pvkeys/shamimanasrinsetu_info/si-bangla-2.json',
        'si-bangla-3' => 'pvkeys/shamimanasrinsetu_info/si-bangla-3.json',
        'si-bangla-4' => 'pvkeys/shamimanasrinsetu_info/si-bangla-4.json',
        'si-bangla-5' => 'pvkeys/shamimanasrinsetu_info/si-bangla-5.json',
        'si-bangla-6' => 'pvkeys/shamimanasrinsetu_info/si-bangla-6.json',
        'si-bangla-7' => 'pvkeys/shamimanasrinsetu_info/si-bangla-7.json',
        'si-bangla-8' => 'pvkeys/shamimanasrinsetu_info/si-bangla-8.json',
        'si-bangla-9' => 'pvkeys/shamimanasrinsetu_info/si-bangla-9.json',
        'si-bangla-10' => 'pvkeys/shamimanasrinsetu_info/si-bangla-10.json',
        
        // islamicapps.storage6@gmail.com
        'islamicapps-storage6-1' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-1.json',
        'islamicapps-storage6-2' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-2.json',
        'islamicapps-storage6-3' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-3.json',
        'islamicapps-storage6-4' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-4.json',
        'islamicapps-storage6-5' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-5.json',
        'islamicapps-storage6-6' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-6.json',
        'islamicapps-storage6-7' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-7.json',
        'islamicapps-storage6-8' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-8.json',
        'islamicapps-storage6-9' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-9.json',
        'islamicapps-storage6-10' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-10.json',
        'islamicapps-storage6-10-1160b' => 'pvkeys/islamicapps_storage6/islamicapps-storage6-11.json',
        
        // islamicapps.storage5@gmail.com for Ramadan Planner
        'islamicapps-storage5-7-da5c4' => 'pvkeys/islamicapps_storage5/islamicapps-storage5-11.json'
    ];

    // Promo URL sets: external_link + feature_image
        // Promo URL sets: external_link + feature_image
    $promoSets = [
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.islamiclibrarybangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/DyS-CUvSHSTdyQI-bWHWEJE2bOiIhvVNqa7G8umDPWAIpsbxxXM2FoNsrDhLKFbX1DyeLHlxpDdgLl-BH-Sh5q4=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafhim.bangla1',
            'feature_image' => 'https://play-lh.googleusercontent.com/w9kEVE44w8RgxRWU1GrXPkXHCQNzKl7OGORWbo4FdGSLAa9W6ELTgtI-olxVcSLpOfI=w240-h480'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafhim.bangla2',
            'feature_image' => 'https://play-lh.googleusercontent.com/eTOoGQb7AnfUi0Rf8TywkZjetLU1q5EChi1q70qO3iQlT-dg0i9FLnv8OpulifsLyQ=w240-h480'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=devsbrain.tafhimulQurangBangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/WJvfD43s6W54WEA4vVQySd8bOEk7lMSVummf_AOcVAXfViHRMyVwuFcxXA2pONV2e682=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bishoyvittikquran',
            'feature_image' => 'https://play-lh.googleusercontent.com/2mhQQjmZ9JbUsMifiY6dQaKEKTFEaeLTEM3xhBPymbm7J7ytOUyvz0a2y3GAdp-oqmM=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsirjalalaine',
            'feature_image' => 'https://play-lh.googleusercontent.com/W6IHVn8Mkk9mUiJa0GuMO6xofZtd-dQ1vQpsLMCnj8Nw7_l_uXPD7WpIByABQtIhFa8h=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.shobdeshobdealquran',
            'feature_image' => 'https://play-lh.googleusercontent.com/5lRTGjs61pWi2UIJHTkbi5vyAWLGpwAwBDbML6zpCmPBJ1LEuGcUZXUCqL16v7ykPWbR=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnkathir',
            'feature_image' => 'https://play-lh.googleusercontent.com/TNFzEbOJs5EneNWEUrQk3weld1_FMpRpS76-s_cWSUQYnUTFXUwekrhva3uPMun54A=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.talimulQuran',
            'feature_image' => 'https://play-lh.googleusercontent.com/UX35gYsJJTVHwt3w9p272vzW6c6LAUHkWWKA7oclltXowIueIiNzhWH9Xc46Cp6qk718=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tazbidshikkha',
            'feature_image' => 'https://play-lh.googleusercontent.com/Qh3L91dH3a5eciDAwRF8up4QzCUldFfbaWanvpkoPelMWJPYBfKWcSiDozhOB3w-Yuw=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.shobdarthealquran',
            'feature_image' => 'https://play-lh.googleusercontent.com/T3Mc2hUNViY-AlE6OflRuxx_lWQPzXehHpz25xWZI3-wZic-ElhP15pF0uBcS1c8BlU=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sonchoyon',
            'feature_image' => 'https://play-lh.googleusercontent.com/wBE5dGaaijygOi8Vv5xy4RAK9NN3D7QE0ulDBJVkAWKRc3NbT07TgG4Be89eK1dY7Bk=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsirtabari',
            'feature_image' => 'https://play-lh.googleusercontent.com/KzEe1WcSOlXkx-wpNWQJ1u2YYE600Y7vBCEjpwFONbAziA5gwvzliz7v67oFnqBPxCI=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranhadithdoa',
            'feature_image' => 'https://play-lh.googleusercontent.com/t18hLCoG4R30pR0JWKW4inqSXFJZZHyMcd0R1o3B_zhH86CNS9garmG1QAvZAjm_7g=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.drasequran',
            'feature_image' => 'https://play-lh.googleusercontent.com/7A0N34qsy6S1DhXp_j2s0uvveUl6GQA8dwDRYtvObjIkVDu6bz6dv-8CdoGIWLm5_A=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.chotoderquranergolpo',
            'feature_image' => 'https://play-lh.googleusercontent.com/KZ3D1sgmTBeOxXXAUzcNJaYqpkjeJXibl5XCuobA6YWX3PCDiRNMNGRxLBOgvvmyAMQ=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranersobdartho',
            'feature_image' => 'https://play-lh.googleusercontent.com/u59UYRmec0_h69L_tLLV8HrQokmRL89n0KXKOwdnOcojy4Kt2BCNTtbyldBeUdDZbfiI=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranergolpo',
            'feature_image' => 'https://play-lh.googleusercontent.com/wufU1uymdzkJhwjyyBYLma6vIQIRVwKCt6mBxRbKKzYhQFnCj8JZeTCur-meED4gM94=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.marefulquranbangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/o6k9fXwDoHkasD-iLWXHKSeIQE3FmwQZozguJIHeHJf2vb-XikUzlEjkOJsaLu--srC8=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bishoyvittikquranayat',
            'feature_image' => 'https://play-lh.googleusercontent.com/BLqPnj7NNcHjrAl9tsIdsfauPbYAtg3UPBjs-6J7PfvF057K7-oml6vpaNw-Vv1fLMpX=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.fizilalilquran.bangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/t1GKM3f4aAN2U2F7Lr_zPLB2zw5xpS3EguwaRh8imdmJBBtqtAD5PfY3Zyu522Jd8nM=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.ampara',
            'feature_image' => 'https://play-lh.googleusercontent.com/UWya6u2mjVaM0bkoniqqYsr1Fy-T-eIrjtxmAifQHz8dB3-BJl7YMvDh4-8eGgtQAq1O=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranerovidhan',
            'feature_image' => 'https://play-lh.googleusercontent.com/GwL3dvBN-5jdFtejLwbC5UzdVFePZqQhLgcqdCuM3MqFNpZ58eHQlRFztU5JIyXW3nE=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=devsbrain.tafsiribnkathiroffline',
            'feature_image' => 'https://play-lh.googleusercontent.com/t83oxscavKNqNmOXlyn__bgyy6PZhWdTLW_HMYknvGc5ox6WOgUbeiyzNUst4JGyT0A=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.darsequran',
            'feature_image' => 'https://play-lh.googleusercontent.com/EiWaCDO1MjeHfdSB8GsVVZgoVf_XYoaG9h2q4gqSJebWhAtzUXRmauDN3xTVdrHs1H4=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.mishkatsharif',
            'feature_image' => 'https://play-lh.googleusercontent.com/kBYbvhmglgnuXsIP0uOv9lmlAnxaF6OdUL2NOCs3_sV2ey1KLAGHD3ah4E-4F35VnA=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.hadithqudsibangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/vDEYLqJndJsgcKmHibH43RBJRvZJbShOwIHGFJ03tHDFx0GEt34b1aIqBwYjGhIZpTDV=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.kitabulfitan',
            'feature_image' => 'https://play-lh.googleusercontent.com/jOeSfhZDlBvpOYXBp_dX9p1DKsIXGzfBWSAcsx_cVpV2siAf4MDniYWnnqukFGYG0uM=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.imamnawawi40hadithBangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/5kpGxYkBIrZc7HHC1dIeYkG9sU0rOT9FhoAUGtgn1EcRaWW3UmoIPKDXBXKGoQhUlQ=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.hadiseralokemanobjibon',
            'feature_image' => 'https://play-lh.googleusercontent.com/8OQKRLqyeZbkdiTQt_WIUlXQR-nCIPqPXJ0aGowwQdl-V1t7FcKiAAYB44edGtywukHK=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bukharisharifbangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/DK3zyiHf4nOiU6ZvBhnnN5uqSBYfo17mLQXjyZEl0wF6eQHG7VvhQIxHKZv-CXGcWPs=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.muslimsharifbangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/xUAgR4rgCQWYzjcjfZHT42n8y808iLBXP3W-h_xeQ0ObfQAi9kLVOn7JY23es-7oFP7v=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tirmidhisharif',
            'feature_image' => 'https://play-lh.googleusercontent.com/s-k44GLJaVpm-dYZ5K7Xc31ZBgcDSJ80d13FLXskhJDnjtvnKWc3X8YYlCiFb9k76w=w240-h480-rw'
        ],
        [
            'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.abudaudsharifbangla',
            'feature_image' => 'https://play-lh.googleusercontent.com/bri7oDkfgi6vA0up5eSrh1JytQf8Sq8hv-PNojl_mM0NGenvB2hXiTKMXWHLx9J5Bg=w240-h480-rw'
        ],

        // … Continue the same pattern for the remaining pairs you listed …
        // For each app:
        // [
        //     'external_link' => 'PLAY_STORE_URL',
        //     'feature_image' => 'IMAGE_URL'
        // ],

    ];


    // Form inputs
    $title        = $_POST['title'];
    $body         = $_POST['body'];
    $featureImage = isset($_POST['feature_image']) ? $_POST['feature_image'] : '';
    $externalLink = isset($_POST['external_link']) ? $_POST['external_link'] : '';
    $topic        = 'notification';  // Firebase topic

    // Override featureImage & externalLink from rotating promo array
    $promo        = getNextPromo($promoSets, 'promo_index_bangla.txt');
    $featureImage = $promo['feature_image'];
    $externalLink = $promo['external_link'];

    // Truncate ONLY for sending notification (title 50 chars, body 500 chars)
    if (function_exists('mb_substr')) {
        $pushTitle = mb_substr($title, 0, 50);
        $pushBody  = mb_substr($body, 0, 500);
    } else {
        $pushTitle = substr($title, 0, 50);
        $pushBody  = substr($body, 0, 500);
    }

    // Custom data for notification (only extra fields; title/body handled in sendFirebaseTopicNotification)
    $customData = [
        'feature_image' => $featureImage,
        'external_link' => $externalLink
    ];

    $action    = $_POST['action']; // "Send Notification"
    $logOutput = "";

    foreach ($projects as $projectId => $authConfigFile) {
        // 1) Save full data to Firebase Realtime Database
        $saveResponse = saveToFirebase($projectId, $title, $body, $featureImage, $externalLink);
        $logOutput   .= "Project: $projectId <br>Save Response: $saveResponse<br>";
        $logOutput   .= "Used external_link: $externalLink<br>Used feature_image: $featureImage<br><br>";

        // 2) Send push notification with truncated title/body (data-only)
        $accessToken          = getAccessToken($authConfigFile);
        $notificationResponse = sendFirebaseTopicNotification(
            $accessToken,
            $projectId,
            $topic,
            $pushTitle,  // 50 chars
            $pushBody,   // 500 chars
            $customData
        );
        $logOutput .= "Project: $projectId <br>Notification Response: <pre>$notificationResponse</pre><br><br>";
    }

    // Display success message with hidden log content
    echo "<html>
        <head>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    background-color: #f4f4f4;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    min-height: 100vh;
                    margin: 0;
                    padding: 10px;
                    box-sizing: border-box;
                }

                .container {
                    text-align: center;
                    background-color: white;
                    padding: 30px;
                    border-radius: 10px;
                    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
                    max-width: 100%;
                    width: 100%;
                    box-sizing: border-box;
                }

                @media(min-width: 768px) {
                    .container {
                        max-width: 600px;
                        padding: 50px;
                    }
                }

                .success-message {
                    font-size: 20px;
                    color: #28a745;
                    margin-bottom: 20px;
                }

                .button {
                    background-color: #007bff;
                    color: white;
                    padding: 10px 20px;
                    font-size: 16px;
                    border: none;
                    border-radius: 5px;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: background-color 0.3s ease;
                }

                .button:hover {
                    background-color: #0056b3;
                }

                .log-section {
                    margin-top: 20px;
                    display: none;
                    max-width: 100%;
                    height: 250px;
                    overflow-y: auto;
                    border: 1px solid #ddd;
                    padding: 15px;
                    background-color: #f9f9f9;
                    text-align: left;
                    white-space: pre-wrap;
                    word-wrap: break-word;
                }

                .toggle-button {
                    background-color: #6c757d;
                    color: white;
                    padding: 8px 16px;
                    font-size: 14px;
                    border: none;
                    border-radius: 4px;
                    cursor: pointer;
                    display: inline-block;
                    margin-top: 10px;
                }

                .toggle-button:hover {
                    background-color: #5a6268;
                }
            </style>
            <script>
                function toggleLogSection() {
                    var logSection = document.getElementById('logSection');
                    if (logSection.style.display === 'none') {
                        logSection.style.display = 'block';
                    } else {
                        logSection.style.display = 'none';
                    }
                }
            </script>
        </head>
        <body>
            <div class='container'>
                <div class='success-message'> Action: $action was processed successfully!</div>
                <button class='toggle-button' onclick='toggleLogSection()'>View Log</button>
                <div class='log-section' id='logSection'>{$logOutput}</div>
                <a class='button' href='index.php'>Back</a>
            </div>
        </body>
    </html>";
}
?>
