<?php

// Helper function for Base64 URL encoding
function base64UrlEncode($text) {
    return str_replace(
        ['+', '/', '='],
        ['-', '_', ''],
        base64_encode($text)
    );
}

function getAccessToken($authConfigFile, $retryCount = 0) {
    $maxRetries = 3;
    $authConfigString = file_get_contents($authConfigFile);

    if (!$authConfigString) {
        die("Failed to read the service account file: $authConfigFile.");
    }

    $authConfig = json_decode($authConfigString);
    if (!$authConfig || !isset($authConfig->private_key)) {
        die("Invalid service account details.");
    }

    $secret = openssl_get_privatekey($authConfig->private_key);
    if (!$secret) {
        die("Failed to load the private key.");
    }

    $header = json_encode(['typ' => 'JWT', 'alg' => 'RS256']);
    $time = time();
    $start = $time - 60;
    $end = $start + 3600; // 1-hour token expiration

    $payload = json_encode([
        "iss"   => $authConfig->client_email,
        "scope" => "https://www.googleapis.com/auth/firebase.messaging",
        "aud"   => "https://oauth2.googleapis.com/token",
        "exp"   => $end,
        "iat"   => $start
    ]);

    $base64UrlHeader  = base64UrlEncode($header);
    $base64UrlPayload = base64UrlEncode($payload);
    openssl_sign($base64UrlHeader . "." . $base64UrlPayload, $signature, $secret, OPENSSL_ALGO_SHA256);
    openssl_free_key($secret);

    $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . base64UrlEncode($signature);

    // Use cURL to get OAuth token
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://oauth2.googleapis.com/token");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=urn:ietf:params:oauth:grant-type:jwt-bearer&assertion=' . $jwt);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/x-www-form-urlencoded"]);
    $responseText = curl_exec($ch);

    if (curl_errno($ch)) {
        echo "Failed to retrieve the token. cURL Error: " . curl_error($ch) . "\n";
        curl_close($ch);
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }

    curl_close($ch);

    $response = json_decode($responseText);

    if (isset($response->access_token)) {
        return $response->access_token;
    } else {
        echo "Failed to get the access token. Response: " . $responseText . "\n";
        if ($retryCount < $maxRetries) {
            echo "Retrying... Attempt: " . ($retryCount + 1) . "\n";
            return getAccessToken($authConfigFile, $retryCount + 1);
        } else {
            die("Maximum retry attempts reached. Failed to get the token.");
        }
    }
}

function sendFirebaseTopicNotification($accessToken, $projectId, $topic, $title, $body, $customData = []) {
    $url = "https://fcm.googleapis.com/v1/projects/$projectId/messages:send";

    $message = [
        "message" => [
            "topic" => $topic,
            "data"  => array_merge($customData, [
                "title"        => $title,   // truncated title
                "body"         => $body,    // truncated body
                "click_action" => "OPEN_SPECIFIC_PAGE",
                "target_page"  => "DetailsActivity",
                "extra_param"  => "example_param_value"
            ])
        ]
    ];

    $messageJson = json_encode($message);

    $headers = [
        "Authorization: Bearer $accessToken",
        "Content-Type: application/json"
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $messageJson);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    // Optional: small timeout so one bad request doesn't hang too long
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);

    $response = curl_exec($ch);

    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        // You can also write this into your own logOutput string if you want.
        return "Failed to send the push notification. cURL error: " . $err;
    }

    curl_close($ch);
    return $response;
}


function saveToFirebase($projectId, $title, $body, $featureImage, $externalLink) {
    $firebaseUrl = "https://$projectId-default-rtdb.firebaseio.com/notifications/english.json";

    $data = [
        'title'         => $title,
        'body'          => $body,
        'feature_image' => $featureImage,
        'external_link' => $externalLink,
        'timestamp'     => time()
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $firebaseUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json"
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);

    $response = curl_exec($ch);

    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        return "Failed to save data to Firebase. cURL error: " . $err;
    }

    curl_close($ch);
    return "Data saved successfully.";
}


/**
 * Promo URL sets: English apps (Play Store + feature image)
 */
$promoSets = [
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsirassadi_ib_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/5NrV2tSzB1yLYRCKUrZ85nwJyXK0HBfsR4LUh7DfUzWyWLBpa51shc4HbcHdkcDmPKY=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsirassadienglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/sdtSHhFCTGRPl_nSvpc2DM-l--IfHYVtgk6f5HRE6Sk4q0DvI6PLYEGWx1uJKT-sVQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsirkurtubi_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/zp-I4ztdzYZ9tQqJOF-wEYFVq_YQsF8h-24MozydD0xhC6fP4ElRw2-dcxJ7HmMirR8=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranbywordenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/Ge3revHuvTsZbXE9mwRU5HxOHBGLoy4XuP9R-suc9Mt5LOM8rGonA5cwweVRBxbffnc=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.lisanulquranenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/DgE3uStd8YAWOJBZuUD85suwxXUGXKlhl1cd5lAAKRiTr7HSwmDw8zkD1MjpjqowtQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnkathir_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/nuByxA15iParcBzidFmJ8A3xuoa9vBNL3LN1t8IhKKmkuju_Qrm9_MvjNhKI9gT6xez7=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.muattaibnmalikenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/ZtRXFLd71ihlcsBwYYGk0N56sYzvAKwcE_4PU6nT4MUEgK2IFjH1PuUtr-UvuC4EpzA=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bukharisharif_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/YmdyXKXjJCSgnEKhuInuQzfBsTMJAm0af3dxug8p43e2lSxJERhBPGiaoEIp8genfA=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sahihmuslimenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/eyXqUxTS_hlOR-st6vH7opUgZxflwmCEC7-T2CwvnQ2oubBVRnRt1dO2R7Mv4Rx2QN0=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnkathirenglishsurabased_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/8lIxbAL4c9RCTtpQqOLt6I0X-pxvTAIdqkw0tMxDDSzWiMP4RbZuzN1JAvY3y1RppXYs=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.prophetcompanions_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/bKabW8UYlm5riGuUohcERTfWo3vTX1rG-TJgzBEcruCBv4nFxqjZRl2CFshDL57ZpEYE=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.jameattirmidhienglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/bKabW8UYlm5riGuUohcERTfWo3vTX1rG-TJgzBEcruCBnbv4nFxqjZRl2CFshDL57ZpEYE=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sunanabudaudenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/kC-c0SwtomB5XchmLwWjtWwg1REP1PRcGz5Zk3KtiB02l0Lw6s1kKm_0gUizaInaiKDK=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sunanibnmajah_english_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/7IvHzHggOzUhaxgqTE53VLwP5rfzhO0eOdNfDbFlUUtj3S-GcSLTTIBKIBhu9uVJv78=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sunanibnnasai_english_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/hjD8hqLerDOlJGDOkgsdACt-1wSaCXAwtMMlK4dnckDdmNEVKBqsAyFtze89WREbjg=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.imamnawawi40hadithenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/3QGDb5AR6V6AiFNbq_rTr_KEqpVDaTjU6etgCwJ3C2l9nO9B-oEoFilqDmRac9juWQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.goldenhadithenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/VItot2GycdOolUCzFYF0r7BPiQUW0aO74bZvzBnDqSuDR_-HQhmYUiQivRMLi3WlPw=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.riadussalehinenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/TBW8vRSdBo7Gq4Yj3eBV0F27LrAYlmHWoFPk327C3ksFdSn3yVIr8cQPVaxeuJpwYco=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.hadithqudsienglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/kC-c0SwtomB5XchmLwWjtWwg1REP1PRcGz5Zk3KtiB02l0Lw6s1kKm_0gUizaInaiKDK=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.hadithcollectionenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/ObvBmh62zyIvH7DKohRHEcT6MtA9q6JI_TJbmarGFvv3cG6ffYnkQJK1gxQzZnGMjl8=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bulugulmaramenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/GKSaFHr_wOELS5sD6kiew7zmkQeloZiOVBSosfmehDFjxCvn-a37Xexu1embtW7Png=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.alluluwalmarjanenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/VKFbNpQ_zBzX2RYNSZLq4wnoVBQ_gNm7t2Idkf1o186F15hJcWK50-kKZRDGlsRQzeJb=w240-h480-rw'
    ],
    [
        // Note: order was image then play-link in your message; fixed here.
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.alaqidahalwasitiyyah_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/28oYyQ-WsciIERiIxLtaRmZbcg2S4wj3LluNbPbvMka-3Tc1XBHoyTwwV4MQY0kgZA=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.albidayawannihaya.english_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/bZ5IsXYGuhTnp7EevG-pGw1iphoxl2Pc_7B94u-CfQiF9Dt2tzRdVF4u9Cl4-3ceta8w=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.atlasofthequran_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/Xy1xvaTXALhS77-B5E21BmIAJdLW3UkQ2InzCmgNoCxPybAEY5RubAbMFsbooIVVQ04x=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.sirahcollection_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/APe6ERP05O6gc6pOrK8ZsQ-9nyPrdKyX_KEtKoIdrYdIfHUj4SL-JtgqPvBxBkHYcQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.arrahequlmakhtum_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/kC-axQ0lvWFG0gLbLyGPzmCan85YNeFT0NSaf1Kwqi9J11g_kSdgTCRtwj_nmwAtMSM=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.islamicstudies_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/u2CtdCeoMw6xXo7Byz-TsVvaANSWOJo0svz_lmSVil7BA-cBJjE-mFb0oL9G3aJUsxOs=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.alqaidaalquraniyyah_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/B9Fq8jyx287peFph3nO0VkAJwRD8k7WM3_Qd6SCO3xHdAnDWgh0170Npa5zw6OfxmVc=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.hisnulmuslimenglish_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/AqM-pouzPuLT2nCVk1U6qEOWnIYZemZqKZMkGREEvcX14LcY0CA8z2hlUcW-9g8NG60=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.fiqhussunnah_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/BjCEsguJXCwFTAjVmyp_7iGLsmlkXgZUdam2bscISk040GhllJe6MVW2W1pHIDKbcVk=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.fatwaislamiyah_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/t5Dl66uyh7o-gwlYpBygcl5T3KwZdtnUA3qCvzVpwT09OWUzxtxZnoXU8QWabvU0wZ5Z=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.kitabut_tauhid_english_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/5XjqT-tBMeVXM6iFKilVgk9eAmyUZ42SzlER_khYVokMTDabMKJsI4aqyLHmt3u00D8=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.greatmsulims_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/KgbOHLKOWo2GkUH5chADNk0ZZPap2N1tF-TwHpCF5TmckJFekTV_VcEY3J_WpNWAsNw=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.greatwomenofislam_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/H1Ozm6I6RO0M-Jj5X2cGTwqTXdBnfYz14U2Fr5GGdsB_7ZTUbSza11SsKaeXMoWcKQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.musliminventions_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/Xu9-NHZb056zxYZmxVvJ86WNGrCm3sOpjq_28Eko0E-xw27vCHc70VYDs6tOOnf6_pQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.bookoftheend_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/5r4142LLdydNjaGHz90btXfeCwOIi5MoeeoEHAAbS8NRxERJfr7z9Yb6iXBLE-lsgQ=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.endoftheworld_english_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/ep2mCr94QvFZYHW0xx6rKNYpqAPHO_-a0XL-4bHUBnRMkB6bNVnq9mO473nFag9PRgDN=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnkathirhindi_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/acs0KZ-W6htu4cU94r2gN7oIWlPWKDpiPSVZw2m51CsYmV_Z0yp286Kom_S6W9jXcAU=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnukatsirlengkap_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/u5yg9tYEHjV8ZP-WOUG3IKzZfKRrpynwVkhy6PvM8DT1KNSPBTHphIjCjkP_i3ou4g=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.quranbywordsurdu_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/wxA-6o3yY_AwwundLdxOZVY61P8e4Pxy33Y3JQMS_R0bbeb4kd0TMQ9V-Pixf0UVcVs=w240-h480-rw'
    ],
    [
        'external_link' => 'https://play.google.com/store/apps/details?id=islamicbooks.tafsiribnkathir.urdu_new',
        'feature_image' => 'https://play-lh.googleusercontent.com/E_CdpMyBHsIgD3xSCLwaRXIOy80YymRRqJUku0srkEyIdE_s2TtSUHc6AdYUMV6RsTw=w240-h480-rw'
    ],
];

/**
 * Get next promo set in rotation, using a simple index file.
 */
function getNextPromoSet(array $promoSets, $indexFile = 'promo_index_english.txt') {
    $count = count($promoSets);
    if ($count === 0) {
        return ['external_link' => '', 'feature_image' => ''];
    }

    $currentIndex = 0;

    if (file_exists($indexFile)) {
        $content = trim(file_get_contents($indexFile));
        if (ctype_digit($content)) {
            $currentIndex = (int)$content;
        }
    }

    $currentIndex = $currentIndex % $count;
    $selected     = $promoSets[$currentIndex];

    $nextIndex = ($currentIndex + 1) % $count;
    file_put_contents($indexFile, (string)$nextIndex);

    return $selected;
}

// Process form data
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // Firebase project configurations
    $projects = [
        //islamicapps.storage1@gmail.com
        'english-storage1-1'      => 'pvkeys/english-storage1-1.json',
        'english-storage1-2'      => 'pvkeys/english-storage1-2.json',
        'english-storage1-2-4b06f'=> 'pvkeys/english-storage1-2-4b06f.json',
        'english-storage1-3'      => 'pvkeys/english-storage1-3.json',
        'english-storage1-4'      => 'pvkeys/english-storage1-4.json',
        'english-storage1-5'      => 'pvkeys/english-storage1-5.json',
        'english-storage1-6'      => 'pvkeys/english-storage1-6.json',
        'english-storage1-7'      => 'pvkeys/english-storage1-7.json',
        'english-storage1-8'      => 'pvkeys/english-storage1-8.json',
        'english-storage1-9'      => 'pvkeys/english-storage1-9.json',
        'english-storage1-10'     => 'pvkeys/english-storage1-10.json',

        //islamicappsstore.info@gmail.com
        'islamic-apps-94b4d'      => 'pvkeys/islamic-apps-94b4d.json',
        'islamic-apps-2'          => 'pvkeys/islamic-apps-2.json',
        'islamic-apps---iasi-3'   => 'pvkeys/islamic-apps---iasi-3.json',
        'islamic-apps---iasi-4'   => 'pvkeys/islamic-apps---iasi-4.json',
        'islamic-apps---iasi-5'   => 'pvkeys/islamic-apps---iasi-5.json',
        'islamic-apps---iasi-6'   => 'pvkeys/islamic-apps---iasi-6.json',
        'islamic-apps---iasi-7'   => 'pvkeys/islamic-apps---iasi-7.json',
        'islamic-apps---iasi-8'   => 'pvkeys/islamic-apps---iasi-8.json',
        'islamic-apps---iasi-9'   => 'pvkeys/islamic-apps---iasi-9.json',
        'islamic-apps---iasi-10'  => 'pvkeys/islamic-apps---iasi-10.json',

        //islamicapps.firebase3@gmail.com // english Apps
        'english-1-alt'           => 'pvkeys/english-1-alt.json',
        'fir-2-alt'               => 'pvkeys/english-2-fir-2-alt.json',
        'english-3-alt'           => 'pvkeys/english-3-alt.json',
        'english-4-alt'           => 'pvkeys/english-4-alt.json',
        'english-5-alt'           => 'pvkeys/english-5-alt.json',
        'english-6-alt'           => 'pvkeys/english-6-alt.json',

        // islamicapps.firebase4@gmail.com // english Apps
        'english-41'              => 'pvkeys/english-41.json',
        'english-42'              => 'pvkeys/english-42.json',
        'arabic-urdu-1e72e'       => 'pvkeys/english-43-arabic-urdu.json',
        'english-44'              => 'pvkeys/english-44.json',
        'english-45'              => 'pvkeys/english-45.json',
        'english-46'              => 'pvkeys/english-46.json',
        'english-47-5e50d'        => 'pvkeys/english-47.json',
        'english-48'              => 'pvkeys/english-48.json',
        'english-49'              => 'pvkeys/english-49.json',
        'english-42-b8622'        => 'pvkeys/english-50-english-42-b8622.json',  // english 50
        'english-48-8868e'        => 'pvkeys/english-51-english-48-8868e.json',  // english 51

        //arifhasnat.business@gmail.com
        'islamic-apps---english1' => 'pvkeys/islamic-apps---english1.json', // english - tafsir ibn kathir english

        //arifhasnat.business2@gmail.com
        'english1-61f90'          => 'pvkeys/english1-61f90.json',
        'english2-ded01'          => 'pvkeys/english2-ded01.json',
        'english3-c3671'          => 'pvkeys/english3-c3671.json',

        //arifhasnat.business3@gmail.com
        'english2-d2a3c'          => 'pvkeys/english2-d2a3c.json',
        'english-13'              => 'pvkeys/english-13.json',
        'english-13-de8e5'        => 'pvkeys/english-13-de8e5.json',

        //islamicapps.storage3@gmail.com
        'islamicapps-storage3-1'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-1.json',
        'islamicapps-storage3-2'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-2.json',
        'islamicapps-storage3-3'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-3.json',
        'islamicapps-storage3-4'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-4.json',
        'islamicapps-storage3-5'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-5.json',
        'islamicapps-storage3-6'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-6.json',
        'islamicapps-storage3-7'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-7.json',
        'islamicapps-storage3-8'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-8.json',
        'islamicapps-storage3-9'  => 'pvkeys/islamicapps_storage3/islamicapps-storage3-9.json',
        'islamicapps-storage3-10' => 'pvkeys/islamicapps_storage3/islamicapps-storage3-10.json',

        //islamicapps.storage4@gmail.com
        'islamicapps-storage4-1'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-1.json',
        'islamicapps-storage4-2'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-2.json',
        'islamicapps-storage4-3'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-3.json',
        'islamicapps-storage4-4'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-4.json',
        'islamicapps-storage4-5'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-5.json',
        'islamicapps-storage4-6'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-6.json',
        'islamicapps-storage4-7'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-7.json',
        'islamicapps-storage4-8'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-8.json',
        'islamicapps-storage4-9'  => 'pvkeys/islamicapps_storage4/islamicapps-storage4-9.json',
        'islamicapps-storage4-10' => 'pvkeys/islamicapps_storage4/islamicapps-storage4-10.json',

        //islamicapps.storage5@gmail.com
        //'islamicapps-storage5-1' => 'pvkeys/islamicapps_storage5/islamicapps-storage5-1.json',
        'islamicapps-storage5-2'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-2.json',
        'islamicapps-storage5-3'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-3.json',
        'islamicapps-storage5-4'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-4.json',
        'islamicapps-storage5-5'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-5.json',
        'islamicapps-storage5-6'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-6.json',
        'islamicapps-storage5-7'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-7.json',
        'islamicapps-storage5-8'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-8.json',
        'islamicapps-storage5-9'  => 'pvkeys/islamicapps_storage5/islamicapps-storage5-9.json',
        'islamicapps-storage5-10' => 'pvkeys/islamicapps_storage5/islamicapps-storage5-10.json',
        'islamicapps-storage5-7-da5c4' => 'pvkeys/islamicapps_storage5/islamicapps-storage5-11.json',
    ];

    // Form inputs
    $title = $_POST['title'];
    $body  = $_POST['body'];
    $topic = 'notification';  // Firebase topic

    // Get next promo (image + external link) from rotation
    $promoSet    = getNextPromoSet($promoSets, 'promo_index_english.txt');
    $featureImage = $promoSet['feature_image'];
    $externalLink = $promoSet['external_link'];

    // Custom data for notification (full text preserved here)
    $customData = [
        'full_title'    => $title,
        'full_body'     => $body,
        'feature_image' => $featureImage,
        'external_link' => $externalLink
    ];

    // Which button (for future use if you add more)
    $action    = $_POST['action'];
    $logOutput = "";

    // Prepare truncated versions for push notification only
    if (function_exists('mb_substr')) {
        $pushTitle = mb_substr($title, 0, 90);
        $pushBody  = mb_substr($body, 0, 990);
    } else {
        $pushTitle = substr($title, 0, 90);
        $pushBody  = substr($body, 0, 990);
    }

    // Single action flow: Save to Firebase first, then send push
    foreach ($projects as $projectId => $authConfigFile) {
        // Save data to Firebase Realtime Database (full text + promo info)
        $saveResponse = saveToFirebase($projectId, $title, $body, $featureImage, $externalLink);
        $logOutput   .= "Project: $projectId <br>Save Response: $saveResponse<br><br>";

        // Get OAuth2 access token
        $accessToken = getAccessToken($authConfigFile);

        // Send push notification with truncated title/body
        $notificationResponse = sendFirebaseTopicNotification(
            $accessToken,
            $projectId,
            $topic,
            $pushTitle,  // 90 chars max
            $pushBody,   // 990 chars max
            $customData
        );
        $logOutput .= "Project: $projectId <br>Notification Response: $notificationResponse<br><br>";
    }

    // Display success message with hidden log content
    echo "<html>
        <head>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    background-color: #f4f4f4;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    min-height: 100vh;
                    margin: 0;
                    padding: 10px;
                    box-sizing: border-box;
                }
                .container {
                    text-align: center;
                    background-color: white;
                    padding: 30px;
                    border-radius: 10px;
                    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
                    max-width: 100%;
                    width: 100%;
                    box-sizing: border-box;
                }
                @media(min-width: 768px) {
                    .container {
                        max-width: 600px;
                        padding: 50px;
                    }
                }
                .success-message {
                    font-size: 20px;
                    color: #28a745;
                    margin-bottom: 20px;
                }
                .button {
                    background-color: #007bff;
                    color: white;
                    padding: 10px 20px;
                    font-size: 16px;
                    border: none;
                    border-radius: 5px;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: background-color 0.3s ease;
                }
                .button:hover {
                    background-color: #0056b3;
                }
                .log-section {
                    margin-top: 20px;
                    display: none;
                    max-width: 100%;
                    height: 250px;
                    overflow-y: auto;
                    border: 1px solid #ddd;
                    padding: 15px;
                    background-color: #f9f9f9;
                    text-align: left;
                    white-space: pre-wrap;
                    word-wrap: break-word;
                }
                .toggle-button {
                    background-color: #6c757d;
                    color: white;
                    padding: 8px 16px;
                    font-size: 14px;
                    border: none;
                    border-radius: 4px;
                    cursor: pointer;
                    display: inline-block;
                    margin-top: 10px;
                }
                .toggle-button:hover {
                    background-color: #5a6268;
                }
            </style>
            <script>
                function toggleLogSection() {
                    var logSection = document.getElementById('logSection');
                    if (logSection.style.display === 'none') {
                        logSection.style.display = 'block';
                    } else {
                        logSection.style.display = 'none';
                    }
                }
            </script>
        </head>
        <body>
            <div class='container'>
                <div class='success-message'> Action: $action was processed successfully!</div>
                <button class='toggle-button' onclick='toggleLogSection()'>View Log</button>
                <div class='log-section' id='logSection'>{$logOutput}</div>
                <a class='button' href='index.php'>Back</a>
            </div>
        </body>
    </html>";
}
?>
